/**
 * Codigos Postales España - Autocompletado de formularios
 * https://api.cerocoma.net
 *
 * Creado por: Fichiando
 * Contacto: info@fichiando.com
 */

(function($) {
    'use strict';

    // Selectores para detectar campos de CP automaticamente
    const CP_SELECTORS = [
        'input[name*="codigo_postal"]',
        'input[name*="postal_code"]',
        'input[name*="postcode"]',
        'input[name*="zipcode"]',
        'input[name*="zip_code"]',
        'input[name*="billing_postcode"]',
        'input[name*="shipping_postcode"]',
        'input[id*="codigo_postal"]',
        'input[id*="postal_code"]',
        'input[id*="postcode"]',
        'input[id*="zip"]',
        'input.cpapi-cp-input',
        '#billing_postcode',
        '#shipping_postcode'
    ];

    // Selectores para campos de municipio/ciudad
    const CITY_SELECTORS = [
        'input[name*="ciudad"]',
        'input[name*="municipio"]',
        'input[name*="localidad"]',
        'input[name*="city"]',
        'input[name*="billing_city"]',
        'input[name*="shipping_city"]',
        'input[id*="ciudad"]',
        'input[id*="municipio"]',
        'input[id*="city"]',
        'input.cpapi-municipio-input',
        '#billing_city',
        '#shipping_city'
    ];

    // Selectores para campos de provincia/estado
    const STATE_SELECTORS = [
        'input[name*="provincia"]',
        'input[name*="state"]',
        'input[name*="region"]',
        'input[name*="billing_state"]',
        'input[name*="shipping_state"]',
        'input[id*="provincia"]',
        'input[id*="state"]',
        'input.cpapi-provincia-input',
        '#billing_state',
        '#shipping_state',
        'select[name*="provincia"]',
        'select[name*="state"]',
        'select[name*="billing_state"]',
        'select[name*="shipping_state"]'
    ];

    // Cache de resultados
    const cache = {};

    // Debounce
    let debounceTimer;

    /**
     * Busca datos por codigo postal
     */
    function buscarCP(cp, callback) {
        // Validar CP
        if (!cp || cp.length < 4) {
            return;
        }

        // Cache
        if (cache[cp]) {
            callback(cache[cp]);
            return;
        }

        // Llamada directa a la API (mas rapido que AJAX de WP)
        const url = cpapi_vars.api_url + '/cp/' + encodeURIComponent(cp);

        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            timeout: 5000,
            success: function(response) {
                if (response && response.success && response.municipios && response.municipios.length > 0) {
                    cache[cp] = response;
                    callback(response);
                }
            },
            error: function() {
                // Fallback: intentar via AJAX de WP
                $.ajax({
                    url: cpapi_vars.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'cpapi_buscar_cp',
                        nonce: cpapi_vars.nonce,
                        cp: cp
                    },
                    success: function(response) {
                        if (response && response.success && response.municipios) {
                            cache[cp] = response;
                            callback(response);
                        }
                    }
                });
            }
        });
    }

    /**
     * Rellena los campos del formulario
     */
    function rellenarCampos($cpInput, data) {
        if (!data || !data.municipios || data.municipios.length === 0) {
            return;
        }

        const muni = data.municipios[0];
        const $form = $cpInput.closest('form');
        const $wrapper = $cpInput.closest('.cpapi-autocompletar-wrapper');

        // Buscar campos por contexto (primero en wrapper, luego en form, luego en document)
        function findField(selectors) {
            let $field;

            // Primero en el wrapper del shortcode
            if ($wrapper.length) {
                const wrapperSelectors = selectors.map(s => s.replace('input', 'input, select'));
                $field = $wrapper.find(wrapperSelectors.join(', ')).first();
                if ($field.length) return $field;
            }

            // Luego en el formulario
            if ($form.length) {
                $field = $form.find(selectors.join(', ')).first();
                if ($field.length) return $field;
            }

            // Finalmente en el documento
            return $(selectors.join(', ')).first();
        }

        // Rellenar municipio/ciudad
        const $cityField = findField(CITY_SELECTORS);
        if ($cityField.length && muni.municipio) {
            if ($cityField.is('select')) {
                // Para selects (como WooCommerce), buscar opcion que coincida
                const $option = $cityField.find('option').filter(function() {
                    return $(this).text().toLowerCase().includes(muni.municipio.toLowerCase());
                });
                if ($option.length) {
                    $cityField.val($option.val()).trigger('change');
                }
            } else {
                $cityField.val(muni.municipio).trigger('change');
            }
        }

        // Rellenar provincia
        const $stateField = findField(STATE_SELECTORS);
        if ($stateField.length && muni.provincia) {
            if ($stateField.is('select')) {
                // Para selects, buscar por texto o valor
                const provinciaLower = muni.provincia.toLowerCase();
                const $option = $stateField.find('option').filter(function() {
                    const text = $(this).text().toLowerCase();
                    const val = $(this).val().toLowerCase();
                    return text.includes(provinciaLower) || val.includes(provinciaLower) ||
                           text.includes(muni.cod_provincia) || val.includes(muni.cod_provincia);
                });
                if ($option.length) {
                    $stateField.val($option.val()).trigger('change');
                }
            } else {
                $stateField.val(muni.provincia).trigger('change');
            }
        }

        // Si hay campo de CCAA en el wrapper
        if ($wrapper.length) {
            const ccaaField = $wrapper.data('campo-ccaa');
            if (ccaaField && muni.ccaa) {
                const $ccaaInput = $wrapper.find('#' + ccaaField + ', [name="' + ccaaField + '"]');
                if ($ccaaInput.length) {
                    $ccaaInput.val(muni.ccaa).trigger('change');
                }
            }
        }

        // Disparar evento personalizado
        $cpInput.trigger('cpapi:completed', [muni, data]);
    }

    /**
     * Inicializa el autocompletado para un campo de CP
     */
    function initCPField($input) {
        if ($input.data('cpapi-init')) {
            return;
        }
        $input.data('cpapi-init', true);

        $input.on('input change', function() {
            const cp = $(this).val().replace(/\D/g, '');

            clearTimeout(debounceTimer);

            if (cp.length >= 4) {
                debounceTimer = setTimeout(function() {
                    buscarCP(cp, function(data) {
                        rellenarCampos($input, data);
                    });
                }, 300);
            }
        });

        // Si ya tiene valor, buscar
        if ($input.val() && $input.val().length >= 4) {
            buscarCP($input.val(), function(data) {
                rellenarCampos($input, data);
            });
        }
    }

    /**
     * Inicializacion
     */
    function init() {
        // Inicializar campos de CP existentes
        $(CP_SELECTORS.join(', ')).each(function() {
            initCPField($(this));
        });

        // Observer para campos dinamicos (AJAX, etc)
        if (typeof MutationObserver !== 'undefined') {
            const observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    $(mutation.addedNodes).find(CP_SELECTORS.join(', ')).each(function() {
                        initCPField($(this));
                    });
                });
            });

            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
        }
    }

    // Iniciar cuando el DOM este listo
    $(document).ready(init);

    // Exponer funciones para uso externo
    window.CPAPI = {
        buscarCP: buscarCP,
        initCPField: initCPField
    };

})(jQuery);
