<?php
/**
 * Plugin Name: Codigos Postales España - API Gratuita
 * Plugin URI: https://api.cerocoma.net
 * Description: Autocompletado de direcciones y festivos locales para formularios WordPress. Usa la API gratuita de api.cerocoma.net con 11.058 codigos postales de España y Andorra.
 * Version: 2.0.0
 * Author: Fichiando
 * Author URI: https://fichiando.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: codigos-postales-api
 *
 * ============================================
 * CREADO POR: El equipo de Fichiando
 * WEB: https://fichiando.com
 * CONTACTO: info@fichiando.com
 * ============================================
 *
 * Este plugin es GRATUITO. Si te resulta util,
 * considera invitarnos a un cafe:
 * - Ko-fi: https://ko-fi.com/FichIAndo
 * - PayPal: https://paypal.me/FichIAndo
 * ============================================
 */

if (!defined('ABSPATH')) {
    exit;
}

define('CPAPI_VERSION', '2.0.0');
define('CPAPI_URL', 'https://api.cerocoma.net/festivos/v1');
define('CPAPI_CACHE_TIME', 86400); // 24 horas

/**
 * Clase principal del plugin
 */
class Codigos_Postales_API {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('widgets_init', array($this, 'register_widget'));
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Shortcodes
        add_shortcode('cp_autocompletar', array($this, 'shortcode_autocompletar'));
        add_shortcode('festivos_locales', array($this, 'shortcode_festivos'));
        add_shortcode('festivos_cp', array($this, 'shortcode_festivos'));
        add_shortcode('festivos_municipio', array($this, 'shortcode_festivos_municipio'));

        // AJAX handlers
        add_action('wp_ajax_cpapi_buscar_cp', array($this, 'ajax_buscar_cp'));
        add_action('wp_ajax_nopriv_cpapi_buscar_cp', array($this, 'ajax_buscar_cp'));
    }

    public function init() {
        load_plugin_textdomain('codigos-postales-api', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    public function enqueue_assets() {
        wp_enqueue_style(
            'cpapi-style',
            plugin_dir_url(__FILE__) . 'css/codigos-postales-api.css',
            array(),
            CPAPI_VERSION
        );

        wp_enqueue_script(
            'cpapi-autocompletar',
            plugin_dir_url(__FILE__) . 'js/autocompletar.js',
            array('jquery'),
            CPAPI_VERSION,
            true
        );

        wp_localize_script('cpapi-autocompletar', 'cpapi_vars', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cpapi_nonce'),
            'api_url' => CPAPI_URL
        ));
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'settings_page_codigos-postales-api') {
            return;
        }
        wp_enqueue_style(
            'cpapi-admin-style',
            plugin_dir_url(__FILE__) . 'css/admin.css',
            array(),
            CPAPI_VERSION
        );
    }

    public function register_widget() {
        register_widget('CPAPI_Festivos_Widget');
    }

    public function add_admin_menu() {
        add_options_page(
            __('Codigos Postales API', 'codigos-postales-api'),
            __('Codigos Postales', 'codigos-postales-api'),
            'manage_options',
            'codigos-postales-api',
            array($this, 'render_admin_page')
        );
    }

    /**
     * Pagina de administracion
     */
    public function render_admin_page() {
        ?>
        <div class="wrap cpapi-admin">
            <h1>
                <span style="font-size: 1.5em;">🇪🇸</span>
                <?php _e('Codigos Postales España - API Gratuita', 'codigos-postales-api'); ?>
            </h1>

            <div class="cpapi-admin-grid">
                <!-- Info principal -->
                <div class="cpapi-card">
                    <h2><?php _e('Sobre este plugin', 'codigos-postales-api'); ?></h2>
                    <p>
                        Este plugin conecta tu WordPress con la API gratuita de
                        <a href="https://api.cerocoma.net" target="_blank">api.cerocoma.net</a>,
                        que contiene:
                    </p>
                    <ul>
                        <li><strong>11.058</strong> codigos postales (España + Andorra)</li>
                        <li><strong>8.131</strong> municipios</li>
                        <li><strong>5.293</strong> municipios con festivos locales</li>
                        <li><strong>6.726</strong> municipios con coordenadas GPS</li>
                    </ul>
                </div>

                <!-- Shortcodes -->
                <div class="cpapi-card">
                    <h2><?php _e('Shortcodes disponibles', 'codigos-postales-api'); ?></h2>

                    <h4>Autocompletar direcciones:</h4>
                    <code>[cp_autocompletar campo_cp="codigo_postal" campo_municipio="ciudad" campo_provincia="provincia"]</code>
                    <p class="description">Autocompleta campos de formulario cuando el usuario escribe un CP.</p>

                    <h4>Mostrar festivos por CP:</h4>
                    <code>[festivos_locales cp="30001"]</code>

                    <h4>Mostrar festivos por INE:</h4>
                    <code>[festivos_locales ine="30030"]</code>

                    <h4>Buscar municipio:</h4>
                    <code>[festivos_municipio nombre="Murcia"]</code>
                </div>

                <!-- Integracion con formularios -->
                <div class="cpapi-card">
                    <h2><?php _e('Integracion con formularios', 'codigos-postales-api'); ?></h2>
                    <p>El plugin detecta automaticamente campos de formulario con estos nombres/IDs:</p>
                    <ul>
                        <li><code>codigo_postal</code>, <code>cp</code>, <code>postal_code</code>, <code>zip</code></li>
                        <li><code>ciudad</code>, <code>municipio</code>, <code>city</code>, <code>localidad</code></li>
                        <li><code>provincia</code>, <code>state</code>, <code>region</code></li>
                    </ul>
                    <p><strong>Compatible con:</strong> Contact Form 7, WPForms, Gravity Forms, WooCommerce Checkout, Elementor Forms.</p>
                </div>

                <!-- Donaciones -->
                <div class="cpapi-card cpapi-donate">
                    <h2>☕ <?php _e('¿Te ha sido util?', 'codigos-postales-api'); ?></h2>
                    <p>
                        Este plugin y la API son <strong>100% gratuitos</strong>.
                        Los datos que otros cobran ~800€/año, nosotros te los regalamos.
                    </p>
                    <p>Si te resulta util, considera invitarnos a un cafe:</p>
                    <div class="cpapi-donate-buttons">
                        <a href="https://ko-fi.com/FichIAndo" target="_blank" class="button button-kofi">
                            ☕ Ko-fi
                        </a>
                        <a href="https://paypal.me/FichIAndo" target="_blank" class="button button-paypal">
                            💰 PayPal
                        </a>
                    </div>
                    <p class="cpapi-author">
                        <strong>Creado por:</strong> El equipo de <a href="https://fichiando.com" target="_blank">Fichiando</a><br>
                        <strong>Contacto:</strong> <a href="mailto:info@fichiando.com">info@fichiando.com</a>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * AJAX handler para buscar CP
     */
    public function ajax_buscar_cp() {
        check_ajax_referer('cpapi_nonce', 'nonce');

        $cp = sanitize_text_field($_POST['cp'] ?? '');
        if (strlen($cp) < 4) {
            wp_send_json_error('CP demasiado corto');
        }

        $data = $this->get_data('cp', $cp);
        wp_send_json($data);
    }

    /**
     * Obtiene datos de la API con cache
     */
    public function get_data($endpoint, $param) {
        $cache_key = 'cpapi_' . md5($endpoint . $param);
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return $cached;
        }

        $url = CPAPI_URL . '/' . $endpoint . '/' . urlencode($param);
        $response = wp_remote_get($url, array(
            'timeout' => 10,
            'headers' => array('Accept' => 'application/json')
        ));

        if (is_wp_error($response)) {
            return array('success' => false, 'error' => $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($data && isset($data['success']) && $data['success']) {
            set_transient($cache_key, $data, CPAPI_CACHE_TIME);
        }

        return $data;
    }

    /**
     * Shortcode [cp_autocompletar]
     */
    public function shortcode_autocompletar($atts) {
        $atts = shortcode_atts(array(
            'campo_cp' => 'codigo_postal',
            'campo_municipio' => 'municipio',
            'campo_provincia' => 'provincia',
            'campo_ccaa' => '',
            'placeholder_cp' => 'Codigo postal',
            'mostrar_campos' => 'true',
        ), $atts, 'cp_autocompletar');

        $mostrar = filter_var($atts['mostrar_campos'], FILTER_VALIDATE_BOOLEAN);

        ob_start();
        ?>
        <div class="cpapi-autocompletar-wrapper"
             data-campo-cp="<?php echo esc_attr($atts['campo_cp']); ?>"
             data-campo-municipio="<?php echo esc_attr($atts['campo_municipio']); ?>"
             data-campo-provincia="<?php echo esc_attr($atts['campo_provincia']); ?>"
             data-campo-ccaa="<?php echo esc_attr($atts['campo_ccaa']); ?>">

            <?php if ($mostrar): ?>
            <div class="cpapi-campos">
                <input type="text"
                       name="<?php echo esc_attr($atts['campo_cp']); ?>"
                       id="<?php echo esc_attr($atts['campo_cp']); ?>"
                       class="cpapi-input cpapi-cp-input"
                       placeholder="<?php echo esc_attr($atts['placeholder_cp']); ?>"
                       maxlength="5"
                       autocomplete="postal-code">

                <input type="text"
                       name="<?php echo esc_attr($atts['campo_municipio']); ?>"
                       id="<?php echo esc_attr($atts['campo_municipio']); ?>"
                       class="cpapi-input cpapi-municipio-input"
                       placeholder="Municipio"
                       readonly>

                <input type="text"
                       name="<?php echo esc_attr($atts['campo_provincia']); ?>"
                       id="<?php echo esc_attr($atts['campo_provincia']); ?>"
                       class="cpapi-input cpapi-provincia-input"
                       placeholder="Provincia"
                       readonly>
            </div>
            <?php endif; ?>

            <div class="cpapi-loading" style="display:none;">
                <span class="cpapi-spinner"></span> Buscando...
            </div>
            <div class="cpapi-error" style="display:none;"></div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Shortcode [festivos_locales cp="30001"] o [festivos_cp cp="30001"]
     */
    public function shortcode_festivos($atts) {
        $atts = shortcode_atts(array(
            'cp' => '',
            'ine' => '',
            'anio' => date('Y'),
            'mostrar_municipio' => 'true',
            'mostrar_provincia' => 'true',
            'estilo' => 'tarjeta',
        ), $atts, 'festivos_locales');

        if (empty($atts['cp']) && empty($atts['ine'])) {
            return '<p class="cpapi-error">' . __('Especifica un codigo postal (cp) o codigo INE (ine).', 'codigos-postales-api') . '</p>';
        }

        if (!empty($atts['ine'])) {
            $data = $this->get_data('ine', $atts['ine']);
            $municipios = isset($data['ine']) ? array($data) : array();
        } else {
            $data = $this->get_data('cp', $atts['cp']);
            $municipios = isset($data['municipios']) ? $data['municipios'] : array();
        }

        if (!$data || !isset($data['success']) || !$data['success']) {
            $error = isset($data['error']) ? $data['error'] : __('Error al obtener los datos.', 'codigos-postales-api');
            return '<p class="cpapi-error">' . esc_html($error) . '</p>';
        }

        if (empty($municipios)) {
            return '<p class="cpapi-info">' . __('No se encontraron datos para esta ubicacion.', 'codigos-postales-api') . '</p>';
        }

        return $this->render_festivos($municipios, $atts);
    }

    /**
     * Shortcode [festivos_municipio nombre="Murcia"]
     */
    public function shortcode_festivos_municipio($atts) {
        $atts = shortcode_atts(array(
            'nombre' => '',
            'anio' => date('Y'),
            'estilo' => 'tarjeta',
        ), $atts, 'festivos_municipio');

        if (empty($atts['nombre'])) {
            return '<p class="cpapi-error">' . __('Especifica el nombre del municipio.', 'codigos-postales-api') . '</p>';
        }

        $cache_key = 'cpapi_buscar_' . md5($atts['nombre']);
        $cached = get_transient($cache_key);

        if ($cached === false) {
            $url = CPAPI_URL . '/buscar?q=' . urlencode($atts['nombre']);
            $response = wp_remote_get($url, array('timeout' => 10));

            if (is_wp_error($response)) {
                return '<p class="cpapi-error">' . esc_html($response->get_error_message()) . '</p>';
            }

            $cached = json_decode(wp_remote_retrieve_body($response), true);
            set_transient($cache_key, $cached, CPAPI_CACHE_TIME);
        }

        if (!$cached || !isset($cached['success']) || !$cached['success']) {
            return '<p class="cpapi-error">' . __('No se encontro el municipio.', 'codigos-postales-api') . '</p>';
        }

        $municipios = isset($cached['resultados']) ? $cached['resultados'] : array();
        if (empty($municipios)) {
            return '<p class="cpapi-info">' . __('No se encontraron resultados.', 'codigos-postales-api') . '</p>';
        }

        return $this->render_festivos($municipios, $atts);
    }

    /**
     * Renderiza los festivos
     */
    private function render_festivos($municipios, $atts) {
        $anio = intval($atts['anio']);
        $estilo = sanitize_text_field($atts['estilo']);
        $mostrar_municipio = filter_var($atts['mostrar_municipio'], FILTER_VALIDATE_BOOLEAN);
        $mostrar_provincia = filter_var($atts['mostrar_provincia'], FILTER_VALIDATE_BOOLEAN);

        ob_start();
        ?>
        <div class="cpapi-container cpapi-estilo-<?php echo esc_attr($estilo); ?>">
            <?php foreach ($municipios as $muni): ?>
                <?php
                $festivos = isset($muni['festivos'][$anio]) ? $muni['festivos'][$anio] : array();
                $nombre = isset($muni['municipio']) ? $muni['municipio'] : '';
                $provincia = isset($muni['provincia']) ? $muni['provincia'] : '';
                ?>
                <div class="cpapi-municipio">
                    <?php if ($mostrar_municipio && $nombre): ?>
                        <h4 class="cpapi-titulo">
                            <?php echo esc_html($nombre); ?>
                            <?php if ($mostrar_provincia && $provincia): ?>
                                <span class="cpapi-provincia">(<?php echo esc_html($provincia); ?>)</span>
                            <?php endif; ?>
                        </h4>
                    <?php endif; ?>

                    <?php if (!empty($festivos)): ?>
                        <div class="cpapi-festivos-lista">
                            <span class="cpapi-anio"><?php echo esc_html($anio); ?></span>
                            <ul>
                                <?php foreach ($festivos as $fecha): ?>
                                    <li class="cpapi-fecha">
                                        <?php echo esc_html($this->format_fecha($fecha)); ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php else: ?>
                        <p class="cpapi-vacio"><?php _e('No hay festivos locales registrados.', 'codigos-postales-api'); ?></p>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>

            <p class="cpapi-creditos">
                <small>
                    Datos de <a href="https://api.cerocoma.net" target="_blank" rel="noopener">api.cerocoma.net</a>
                    · Creado por <a href="https://fichiando.com" target="_blank" rel="noopener">Fichiando</a>
                </small>
            </p>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Formatea fecha DD/MM a formato legible
     */
    private function format_fecha($fecha) {
        $meses = array(
            '01' => 'Enero', '02' => 'Febrero', '03' => 'Marzo',
            '04' => 'Abril', '05' => 'Mayo', '06' => 'Junio',
            '07' => 'Julio', '08' => 'Agosto', '09' => 'Septiembre',
            '10' => 'Octubre', '11' => 'Noviembre', '12' => 'Diciembre'
        );

        $partes = explode('/', $fecha);
        if (count($partes) === 2) {
            $dia = intval($partes[0]);
            $mes = isset($meses[$partes[1]]) ? $meses[$partes[1]] : $partes[1];
            return $dia . ' de ' . $mes;
        }

        return $fecha;
    }

    /**
     * Limpia cache
     */
    public static function clear_cache() {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_cpapi_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_cpapi_%'");
    }
}

/**
 * Widget de Festivos
 */
class CPAPI_Festivos_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'cpapi_festivos_widget',
            __('Festivos Locales', 'codigos-postales-api'),
            array('description' => __('Muestra los festivos locales de un municipio.', 'codigos-postales-api'))
        );
    }

    public function widget($args, $instance) {
        echo $args['before_widget'];

        $title = !empty($instance['title']) ? $instance['title'] : __('Festivos Locales', 'codigos-postales-api');
        echo $args['before_title'] . apply_filters('widget_title', $title) . $args['after_title'];

        $cp = !empty($instance['cp']) ? $instance['cp'] : '';
        $anio = !empty($instance['anio']) ? $instance['anio'] : date('Y');

        if ($cp) {
            echo do_shortcode('[festivos_locales cp="' . esc_attr($cp) . '" anio="' . esc_attr($anio) . '" estilo="minimo"]');
        } else {
            echo '<p>' . __('Configura el codigo postal en el widget.', 'codigos-postales-api') . '</p>';
        }

        echo $args['after_widget'];
    }

    public function form($instance) {
        $title = !empty($instance['title']) ? $instance['title'] : __('Festivos Locales', 'codigos-postales-api');
        $cp = !empty($instance['cp']) ? $instance['cp'] : '';
        $anio = !empty($instance['anio']) ? $instance['anio'] : date('Y');
        ?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>"><?php _e('Titulo:', 'codigos-postales-api'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>"
                   name="<?php echo esc_attr($this->get_field_name('title')); ?>"
                   type="text" value="<?php echo esc_attr($title); ?>">
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('cp')); ?>"><?php _e('Codigo Postal:', 'codigos-postales-api'); ?></label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('cp')); ?>"
                   name="<?php echo esc_attr($this->get_field_name('cp')); ?>"
                   type="text" value="<?php echo esc_attr($cp); ?>"
                   placeholder="30001" maxlength="5">
        </p>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('anio')); ?>"><?php _e('Año:', 'codigos-postales-api'); ?></label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('anio')); ?>"
                    name="<?php echo esc_attr($this->get_field_name('anio')); ?>">
                <?php for ($y = date('Y'); $y <= date('Y') + 1; $y++): ?>
                    <option value="<?php echo $y; ?>" <?php selected($anio, $y); ?>><?php echo $y; ?></option>
                <?php endfor; ?>
            </select>
        </p>
        <?php
    }

    public function update($new_instance, $old_instance) {
        $instance = array();
        $instance['title'] = sanitize_text_field($new_instance['title']);
        $instance['cp'] = sanitize_text_field($new_instance['cp']);
        $instance['anio'] = intval($new_instance['anio']);
        return $instance;
    }
}

// Inicializar plugin
Codigos_Postales_API::get_instance();

// Hooks de activacion/desactivacion
register_activation_hook(__FILE__, function() {
    // Nada que hacer por ahora
});

register_deactivation_hook(__FILE__, function() {
    Codigos_Postales_API::clear_cache();
});
